// GoGOST -- Pure Go GOST cryptographic functions library
// Copyright (C) 2015-2021 Sergey Matveev <stargrave@stargrave.org>
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, version 3 of the License.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

package gost34112012

import (
	"bytes"
	"crypto/hmac"
	"hash"
	"testing"
)

func hash256() hash.Hash {
	return New(32)
}

func hash512() hash.Hash {
	return New(64)
}

// HMAC test vectors taken from
// http://tc26.ru/methods/recommendation/%D0%A2%D0%9A26%D0%90%D0%9B%D0%93.pdf test vectors
func TestHMACVectors(t *testing.T) {
	t.Run("256", func(t *testing.T) {
		h := hmac.New(hash256, []byte{
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		})
		h.Write([]byte{
			0x01, 0x26, 0xbd, 0xb8, 0x78, 0x00, 0xaf, 0x21,
			0x43, 0x41, 0x45, 0x65, 0x63, 0x78, 0x01, 0x00,
		})
		if bytes.Compare(h.Sum(nil), []byte{
			0xa1, 0xaa, 0x5f, 0x7d, 0xe4, 0x02, 0xd7, 0xb3,
			0xd3, 0x23, 0xf2, 0x99, 0x1c, 0x8d, 0x45, 0x34,
			0x01, 0x31, 0x37, 0x01, 0x0a, 0x83, 0x75, 0x4f,
			0xd0, 0xaf, 0x6d, 0x7c, 0xd4, 0x92, 0x2e, 0xd9,
		}) != 0 {
			t.FailNow()
		}
	})

	t.Run("512", func(t *testing.T) {
		h := hmac.New(hash512, []byte{
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		})
		h.Write([]byte{
			0x01, 0x26, 0xbd, 0xb8, 0x78, 0x00, 0xaf, 0x21,
			0x43, 0x41, 0x45, 0x65, 0x63, 0x78, 0x01, 0x00,
		})
		if bytes.Compare(h.Sum(nil), []byte{
			0xa5, 0x9b, 0xab, 0x22, 0xec, 0xae, 0x19, 0xc6,
			0x5f, 0xbd, 0xe6, 0xe5, 0xf4, 0xe9, 0xf5, 0xd8,
			0x54, 0x9d, 0x31, 0xf0, 0x37, 0xf9, 0xdf, 0x9b,
			0x90, 0x55, 0x00, 0xe1, 0x71, 0x92, 0x3a, 0x77,
			0x3d, 0x5f, 0x15, 0x30, 0xf2, 0xed, 0x7e, 0x96,
			0x4c, 0xb2, 0xee, 0xdc, 0x29, 0xe9, 0xad, 0x2f,
			0x3a, 0xfe, 0x93, 0xb2, 0x81, 0x4f, 0x79, 0xf5,
			0x00, 0x0f, 0xfc, 0x03, 0x66, 0xc2, 0x51, 0xe6,
		}) != 0 {
			t.FailNow()
		}
	})
}
