// GoGOST -- Pure Go GOST cryptographic functions library
// Copyright (C) 2015-2021 Sergey Matveev <stargrave@stargrave.org>
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, version 3 of the License.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

package gost34112012256

import (
	"bytes"
	"testing"
)

func TestTLSTreeMagmaCTROMAC(t *testing.T) {
	tt := NewTLSTree(TLSGOSTR341112256WithMagmaCTROMAC, []byte{
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	})
	t.Run("0", func(t *testing.T) {
		got := tt.Derive(0)
		if bytes.Compare(got, []byte{
			0x50, 0x76, 0x42, 0xD9, 0x58, 0xC5, 0x20, 0xC6,
			0xD7, 0xEE, 0xF5, 0xCA, 0x8A, 0x53, 0x16, 0xD4,
			0xF3, 0x4B, 0x85, 0x5D, 0x2D, 0xD4, 0xBC, 0xBF,
			0x4E, 0x5B, 0xF0, 0xFF, 0x64, 0x1A, 0x19, 0xFF,
		}) != 0 {
			t.FailNow()
		}
		if bytes.Compare(got, tt.Derive(1)) != 0 {
			t.FailNow()
		}
	})
	t.Run("4095", func(t *testing.T) {
		if bytes.Compare(tt.Derive(4095), []byte{
			0x50, 0x76, 0x42, 0xD9, 0x58, 0xC5, 0x20, 0xC6,
			0xD7, 0xEE, 0xF5, 0xCA, 0x8A, 0x53, 0x16, 0xD4,
			0xF3, 0x4B, 0x85, 0x5D, 0x2D, 0xD4, 0xBC, 0xBF,
			0x4E, 0x5B, 0xF0, 0xFF, 0x64, 0x1A, 0x19, 0xFF,
		}) != 0 {
			t.FailNow()
		}
	})
	t.Run("4096", func(t *testing.T) {
		if bytes.Compare(tt.Derive(4096), []byte{
			0x71, 0x86, 0x93, 0xEC, 0xD3, 0xF1, 0xDC, 0xE2,
			0x11, 0x93, 0xBD, 0x56, 0x40, 0x6A, 0xA4, 0xE2,
			0x09, 0x0F, 0xE1, 0xC8, 0x7A, 0x5D, 0x5D, 0x7A,
			0x8C, 0xE6, 0xE0, 0xD8, 0x28, 0xA4, 0x39, 0x15,
		}) != 0 {
			t.FailNow()
		}
	})
	t.Run("33554431", func(t *testing.T) {
		if bytes.Compare(tt.Derive(33554431), []byte{
			0xCD, 0x4E, 0x6F, 0x09, 0x94, 0xC4, 0xCA, 0x17,
			0xA9, 0xAD, 0xE5, 0x3D, 0x69, 0x20, 0x49, 0x77,
			0x57, 0x87, 0x3B, 0xC0, 0x8D, 0x4F, 0x98, 0x06,
			0xE3, 0xC9, 0x99, 0xA7, 0xE4, 0xAE, 0x70, 0xE0,
		}) != 0 {
			t.FailNow()
		}
	})
	t.Run("33554432", func(t *testing.T) {
		if bytes.Compare(tt.Derive(33554432), []byte{
			0x4E, 0xAB, 0x0F, 0x8A, 0x28, 0x2D, 0xE5, 0x78,
			0x85, 0x7B, 0xE0, 0x15, 0xD1, 0x32, 0xCD, 0xFD,
			0x88, 0xF0, 0xEB, 0xD9, 0x9D, 0x10, 0xCC, 0xA3,
			0xB6, 0xD9, 0x11, 0x4E, 0x37, 0x56, 0x5D, 0xE7,
		}) != 0 {
			t.FailNow()
		}
	})
	t.Run("274877906943", func(t *testing.T) {
		if bytes.Compare(tt.Derive(274877906943), []byte{
			0x37, 0x06, 0xB1, 0xBE, 0xE6, 0xF3, 0xBE, 0xDB,
			0x74, 0x66, 0x4E, 0x1C, 0x43, 0x40, 0xB5, 0xE1,
			0x81, 0x5F, 0xFA, 0x87, 0xEA, 0x80, 0xF4, 0x0E,
			0xAF, 0xCC, 0x28, 0xEF, 0x30, 0xDD, 0xF3, 0xBB,
		}) != 0 {
			t.FailNow()
		}
	})
	t.Run("274877906944", func(t *testing.T) {
		if bytes.Compare(tt.Derive(274877906944), []byte{
			0x9B, 0xE5, 0x7C, 0xB5, 0x2F, 0x46, 0xB6, 0x0C,
			0x68, 0x40, 0xCD, 0x4E, 0x95, 0x9A, 0x16, 0x8A,
			0x74, 0x82, 0x2C, 0x1D, 0x00, 0x6A, 0xC7, 0x46,
			0xDA, 0xBB, 0xB9, 0x84, 0x8E, 0xB0, 0xA5, 0x8A,
		}) != 0 {
			t.FailNow()
		}
	})
}

func TestTLSTreeKuznyechikCTROMAC(t *testing.T) {
	tt := NewTLSTree(TLSGOSTR341112256WithKuznyechikCTROMAC, []byte{
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	})
	t.Run("0", func(t *testing.T) {
		got := tt.Derive(0)
		if bytes.Compare(got, []byte{
			0x50, 0x76, 0x42, 0xD9, 0x58, 0xC5, 0x20, 0xC6,
			0xD7, 0xEE, 0xF5, 0xCA, 0x8A, 0x53, 0x16, 0xD4,
			0xF3, 0x4B, 0x85, 0x5D, 0x2D, 0xD4, 0xBC, 0xBF,
			0x4E, 0x5B, 0xF0, 0xFF, 0x64, 0x1A, 0x19, 0xFF,
		}) != 0 {
			t.FailNow()
		}
		if bytes.Compare(got, tt.Derive(1)) != 0 {
			t.FailNow()
		}
	})
	t.Run("63", func(t *testing.T) {
		if bytes.Compare(tt.Derive(63), []byte{
			0x50, 0x76, 0x42, 0xD9, 0x58, 0xC5, 0x20, 0xC6,
			0xD7, 0xEE, 0xF5, 0xCA, 0x8A, 0x53, 0x16, 0xD4,
			0xF3, 0x4B, 0x85, 0x5D, 0x2D, 0xD4, 0xBC, 0xBF,
			0x4E, 0x5B, 0xF0, 0xFF, 0x64, 0x1A, 0x19, 0xFF,
		}) != 0 {
			t.FailNow()
		}
	})
	t.Run("64", func(t *testing.T) {
		if bytes.Compare(tt.Derive(64), []byte{
			0xC1, 0x9B, 0x63, 0x9F, 0x4B, 0xEA, 0x78, 0x8C,
			0x1C, 0x59, 0xB4, 0xC8, 0x87, 0xDB, 0x5B, 0x07,
			0xC1, 0x91, 0x19, 0x10, 0x18, 0x68, 0xDA, 0xB8,
			0x9A, 0x8D, 0x93, 0x61, 0xB2, 0xF0, 0x10, 0xF3,
		}) != 0 {
			t.FailNow()
		}
	})
	t.Run("524287", func(t *testing.T) {
		if bytes.Compare(tt.Derive(524287), []byte{
			0x92, 0x30, 0x3E, 0xB5, 0x61, 0x56, 0x88, 0x54,
			0xE3, 0x3E, 0x4F, 0xE0, 0x97, 0xA9, 0x95, 0x99,
			0x17, 0x9F, 0x5B, 0x90, 0x94, 0xAE, 0x34, 0x79,
			0xE6, 0x1C, 0x43, 0x69, 0x3A, 0x3F, 0x0A, 0x06,
		}) != 0 {
			t.FailNow()
		}
	})
	t.Run("524288", func(t *testing.T) {
		if bytes.Compare(tt.Derive(524288), []byte{
			0xE8, 0x55, 0xA0, 0xE2, 0xCB, 0xDD, 0x68, 0xC1,
			0x13, 0x7C, 0xEF, 0x3E, 0x80, 0x1E, 0x0B, 0xFF,
			0x68, 0x62, 0x8C, 0x36, 0x43, 0x68, 0x27, 0x6D,
			0x0C, 0xB8, 0x7E, 0xB5, 0x6E, 0x94, 0xEF, 0x42,
		}) != 0 {
			t.FailNow()
		}
	})
	t.Run("4294967295", func(t *testing.T) {
		if bytes.Compare(tt.Derive(4294967295), []byte{
			0x93, 0xD3, 0xCA, 0xE9, 0x5A, 0x55, 0xB7, 0x1A,
			0xA3, 0xB9, 0xA7, 0xDD, 0xF9, 0x9A, 0x6A, 0xAC,
			0x3F, 0xDA, 0x17, 0x2A, 0x79, 0x60, 0x58, 0x04,
			0xA9, 0xC9, 0xFC, 0x6E, 0x84, 0x8A, 0xF1, 0xAA,
		}) != 0 {
			t.FailNow()
		}
	})
	t.Run("4294967296", func(t *testing.T) {
		if bytes.Compare(tt.Derive(4294967296), []byte{
			0x7F, 0xFB, 0x1A, 0xD7, 0xE5, 0x7B, 0x70, 0xBE,
			0x10, 0x96, 0x31, 0xD2, 0x71, 0x92, 0x98, 0xB9,
			0x7D, 0xEE, 0x3B, 0x00, 0x8D, 0x86, 0xF8, 0x3D,
			0xAA, 0xF6, 0x2A, 0x4E, 0xA5, 0xB7, 0xAA, 0xFD,
		}) != 0 {
			t.FailNow()
		}
	})
}
