// GoGOST -- Pure Go GOST cryptographic functions library
// Copyright (C) 2015-2021 Sergey Matveev <stargrave@stargrave.org>
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, version 3 of the License.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

package gost28147

import (
	"bytes"
	"crypto/cipher"
	"crypto/rand"
	"testing"
	"testing/quick"
)

func TestCTRGCL3Vector(t *testing.T) {
	sbox := &SboxIdGost2814789TestParamSet
	key := []byte{
		0x04, 0x75, 0xf6, 0xe0, 0x50, 0x38, 0xfb, 0xfa,
		0xd2, 0xc7, 0xc3, 0x90, 0xed, 0xb3, 0xca, 0x3d,
		0x15, 0x47, 0x12, 0x42, 0x91, 0xae, 0x1e, 0x8a,
		0x2f, 0x79, 0xcd, 0x9e, 0xd2, 0xbc, 0xef, 0xbd,
	}
	plaintext := []byte{
		0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01, 0x00,
		0x0f, 0x0e, 0x0d, 0x0c, 0x0b, 0x0a, 0x09, 0x08,
		0x17, 0x16, 0x15, 0x14, 0x13, 0x12, 0x11, 0x10,
		0x1f, 0x1e, 0x1d, 0x1c, 0x1b, 0x1a, 0x19, 0x18,
		0x27, 0x26, 0x25, 0x24, 0x23, 0x22, 0x21, 0x20,
		0x2f, 0x2e, 0x2d, 0x2c, 0x2b, 0x2a, 0x29, 0x28,
		0x37, 0x36, 0x35, 0x34, 0x33, 0x32, 0x31, 0x30,
		0x3f, 0x3e, 0x3d, 0x3c, 0x3b, 0x3a, 0x39, 0x38,
		0x47, 0x46, 0x45, 0x44, 0x43, 0x42, 0x41, 0x40,
		0x4f, 0x4e, 0x4d, 0x4c, 0x4b, 0x4a, 0x49, 0x48,
		0x57, 0x56, 0x55, 0x54, 0x53, 0x52, 0x51, 0x50,
		0x5f, 0x5e, 0x5d, 0x5c, 0x5b, 0x5a, 0x59, 0x58,
		0x67, 0x66, 0x65, 0x64, 0x63, 0x62, 0x61, 0x60,
		0x6f, 0x6e, 0x6d, 0x6c, 0x6b, 0x6a, 0x69, 0x68,
		0x77, 0x76, 0x75, 0x74, 0x73, 0x72, 0x71, 0x70,
		0x7f, 0x7e, 0x7d, 0x7c, 0x7b, 0x7a, 0x79, 0x78,
		0x87, 0x86, 0x85, 0x84, 0x83, 0x82, 0x81, 0x80,
		0x8f, 0x8e, 0x8d, 0x8c, 0x8b, 0x8a, 0x89, 0x88,
		0x97, 0x96, 0x95, 0x94, 0x93, 0x92, 0x91, 0x90,
		0x9f, 0x9e, 0x9d, 0x9c, 0x9b, 0x9a, 0x99, 0x98,
		0xa7, 0xa6, 0xa5, 0xa4, 0xa3, 0xa2, 0xa1, 0xa0,
		0xaf, 0xae, 0xad, 0xac, 0xab, 0xaa, 0xa9, 0xa8,
		0xb7, 0xb6, 0xb5, 0xb4, 0xb3, 0xb2, 0xb1, 0xb0,
		0xbf, 0xbe, 0xbd, 0xbc, 0xbb, 0xba, 0xb9, 0xb8,
		0xc7, 0xc6, 0xc5, 0xc4, 0xc3, 0xc2, 0xc1, 0xc0,
		0xcf, 0xce, 0xcd, 0xcc, 0xcb, 0xca, 0xc9, 0xc8,
		0xd7, 0xd6, 0xd5, 0xd4, 0xd3, 0xd2, 0xd1, 0xd0,
		0xdf, 0xde, 0xdd, 0xdc, 0xdb, 0xda, 0xd9, 0xd8,
		0xe7, 0xe6, 0xe5, 0xe4, 0xe3, 0xe2, 0xe1, 0xe0,
		0xef, 0xee, 0xed, 0xec, 0xeb, 0xea, 0xe9, 0xe8,
		0xf7, 0xf6, 0xf5, 0xf4, 0xf3, 0xf2, 0xf1, 0xf0,
		0xff, 0xfe, 0xfd, 0xfc, 0xfb,
	}
	ciphertext := []byte{
		0x4a, 0x5e, 0x37, 0x6c, 0xa1, 0x12, 0xd3, 0x55,
		0x09, 0x13, 0x1a, 0x21, 0xac, 0xfb, 0xb2, 0x1e,
		0x8c, 0x24, 0x9b, 0x57, 0x20, 0x68, 0x46, 0xd5,
		0x23, 0x2a, 0x26, 0x35, 0x12, 0x56, 0x5c, 0x69,
		0x2a, 0x2f, 0xd1, 0xab, 0xbd, 0x45, 0xdc, 0x3a,
		0x1a, 0xa4, 0x57, 0x64, 0xd5, 0xe4, 0x69, 0x6d,
		0xb4, 0x8b, 0xf1, 0x54, 0x78, 0x3b, 0x10, 0x8f,
		0x7a, 0x4b, 0x32, 0xe0, 0xe8, 0x4c, 0xbf, 0x03,
		0x24, 0x37, 0x95, 0x6a, 0x55, 0xa8, 0xce, 0x6f,
		0x95, 0x62, 0x12, 0xf6, 0x79, 0xe6, 0xf0, 0x1b,
		0x86, 0xef, 0x36, 0x36, 0x05, 0xd8, 0x6f, 0x10,
		0xa1, 0x41, 0x05, 0x07, 0xf8, 0xfa, 0xa4, 0x0b,
		0x17, 0x2c, 0x71, 0xbc, 0x8b, 0xcb, 0xcf, 0x3d,
		0x74, 0x18, 0x32, 0x0b, 0x1c, 0xd2, 0x9e, 0x75,
		0xba, 0x3e, 0x61, 0xe1, 0x61, 0x96, 0xd0, 0xee,
		0x8f, 0xf2, 0x9a, 0x5e, 0xb7, 0x7a, 0x15, 0xaa,
		0x4e, 0x1e, 0x77, 0x7c, 0x99, 0xe1, 0x41, 0x13,
		0xf4, 0x60, 0x39, 0x46, 0x4c, 0x35, 0xde, 0x95,
		0xcc, 0x4f, 0xd5, 0xaf, 0xd1, 0x4d, 0x84, 0x1a,
		0x45, 0xc7, 0x2a, 0xf2, 0x2c, 0xc0, 0xb7, 0x94,
		0xa3, 0x08, 0xb9, 0x12, 0x96, 0xb5, 0x97, 0x99,
		0x3a, 0xb7, 0x0c, 0x14, 0x56, 0xb9, 0xcb, 0x49,
		0x44, 0xa9, 0x93, 0xa9, 0xfb, 0x19, 0x10, 0x8c,
		0x6a, 0x68, 0xe8, 0x7b, 0x06, 0x57, 0xf0, 0xef,
		0x88, 0x44, 0xa6, 0xd2, 0x98, 0xbe, 0xd4, 0x07,
		0x41, 0x37, 0x45, 0xa6, 0x71, 0x36, 0x76, 0x69,
		0x4b, 0x75, 0x15, 0x33, 0x90, 0x29, 0x6e, 0x33,
		0xcb, 0x96, 0x39, 0x78, 0x19, 0x2e, 0x96, 0xf3,
		0x49, 0x4c, 0x89, 0x3d, 0xa1, 0x86, 0x82, 0x00,
		0xce, 0xbd, 0x54, 0x29, 0x65, 0x00, 0x1d, 0x16,
		0x13, 0xc3, 0xfe, 0x1f, 0x8c, 0x55, 0x63, 0x09,
		0x1f, 0xcd, 0xd4, 0x28, 0xca,
	}
	iv := []byte{0x02, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01}
	c := NewCipher(key, sbox)
	ctr := c.NewCTR(iv)
	tmp := make([]byte, len(plaintext))
	ctr.XORKeyStream(tmp, plaintext)
	if bytes.Compare(tmp, ciphertext) != 0 {
		t.Fatal("encryption failed")
	}
	ctr = c.NewCTR(iv)
	ctr.XORKeyStream(tmp, tmp)
	if bytes.Compare(tmp, plaintext) != 0 {
		t.Fatal("decryption failed")
	}
}

func TestCTRGCL2Vector(t *testing.T) {
	sbox := &SboxIdGost2814789TestParamSet
	key := []byte{
		0xfc, 0x7a, 0xd2, 0x88, 0x6f, 0x45, 0x5b, 0x50,
		0xd2, 0x90, 0x08, 0xfa, 0x62, 0x2b, 0x57, 0xd5,
		0xc6, 0x5b, 0x3c, 0x63, 0x72, 0x02, 0x02, 0x57,
		0x99, 0xca, 0xdf, 0x07, 0x68, 0x51, 0x9e, 0x8a,
	}
	plaintext := []byte{
		0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01, 0x00,
		0x0f, 0x0e, 0x0d, 0x0c, 0x0b, 0x0a, 0x09, 0x08,
		0x17, 0x16, 0x15, 0x14, 0x13, 0x12, 0x11, 0x10,
		0x1f, 0x1e, 0x1d, 0x1c, 0x1b, 0x1a, 0x19, 0x18,
		0x27, 0x26, 0x25, 0x24, 0x23, 0x22, 0x21, 0x20,
		0x2f, 0x2e, 0x2d, 0x2c, 0x2b, 0x2a, 0x29, 0x28,
		0xff, 0xfe, 0xfd, 0xfc, 0xfb,
	}
	ciphertext := []byte{
		0xd0, 0xbe, 0x60, 0x1a, 0x2c, 0xf1, 0x90, 0x26,
		0x9b, 0x7b, 0x23, 0xb4, 0xd2, 0xcc, 0xe1, 0x15,
		0xf6, 0x05, 0x57, 0x28, 0x88, 0x75, 0xeb, 0x1e,
		0xd3, 0x62, 0xdc, 0xda, 0x9b, 0x62, 0xee, 0x9a,
		0x57, 0x87, 0x8a, 0xf1, 0x82, 0x37, 0x9c, 0x7f,
		0x13, 0xcc, 0x55, 0x38, 0xb5, 0x63, 0x32, 0xc5,
		0x23, 0xa4, 0xcb, 0x7d, 0x51,
	}
	var iv [8]byte
	c := NewCipher(key, sbox)
	ctr := c.NewCTR(iv[:])
	tmp := make([]byte, len(plaintext))
	ctr.XORKeyStream(tmp, plaintext)
	if bytes.Compare(tmp, ciphertext) != 0 {
		t.Fatal("encryption failed")
	}
	ctr = c.NewCTR(iv[:])
	ctr.XORKeyStream(tmp, tmp)
	if bytes.Compare(tmp, plaintext) != 0 {
		t.Fatal("decryption failed")
	}
}

func TestCTRRandom(t *testing.T) {
	f := func(key [KeySize]byte, iv [BlockSize]byte, pt []byte) bool {
		if len(pt) == 0 {
			return true
		}
		c := NewCipher(key[:], SboxDefault)
		tmp := make([]byte, len(pt))
		ctr := c.NewCTR(iv[:])
		ctr.XORKeyStream(tmp, pt[:])
		ctr = c.NewCTR(iv[:])
		ctr.XORKeyStream(tmp, tmp)
		return bytes.Compare(tmp, pt) == 0
	}
	if err := quick.Check(f, nil); err != nil {
		t.Error(err)
	}
}
func TestCTRInterface(t *testing.T) {
	var key [KeySize]byte
	var iv [8]byte
	c := NewCipher(key[:], SboxDefault)
	var _ cipher.Stream = c.NewCTR(iv[:])
}

func BenchmarkCTR(b *testing.B) {
	key := make([]byte, KeySize)
	iv := make([]byte, BlockSize)
	rand.Read(key)
	rand.Read(iv)
	dst := make([]byte, BlockSize)
	src := make([]byte, BlockSize)
	rand.Read(src)
	c := NewCipher(key, SboxDefault)
	ctr := c.NewCTR(iv)
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		ctr.XORKeyStream(dst, src)
	}
}
